classdef TopoARTC < TopoART
    properties
        fileFormatVersion {mustBeNumeric}
        nu {mustBeInteger, mustBePositive}
    end

    methods (Access = protected)
        function colorIndex = GetClassColor(tac, moduleIndex, nodeIndex)
            arguments
                tac TopoARTC
                moduleIndex {mustBeInteger, mustBePositive}
                nodeIndex {mustBeInteger, mustBePositive}
            end

            colorIndex = tac.modules(moduleIndex).nodes(nodeIndex).classID - tac.modules(moduleIndex).minClassID + 1;
        end

        function colorNum = GetClassColorNum(tac, moduleIndex)
            arguments
                tac TopoARTC
                moduleIndex {mustBeInteger, mustBePositive}
            end

            colorNum = max(tac.modules(moduleIndex).maxClassID - tac.modules(moduleIndex).minClassID + 1, 2);
        end

        function Init(tac)
            tac.shortName = 'TAC';
            tac.longName = 'TopoART-C';
        end

        function file = OpenFile(tac, path)
            arguments
                tac TopoARTC
                path char {mustBeFile}
            end

            disp(['Load ' tac.longName ' network: ' path])

            file = fopen(path,'r');
            assert(file ~= -1, 'File not found')

            fgetl(file);
            l = fgetl(file);
            assert(strcmp(l, '*       TopoART-C network       *') ~= 0, 'No TopoART-C text file')    
            l = fgetl(file);
            tac.LibTopoART_version = sscanf(l(8:length(l)),'LibTopoART (v%f)');
            fprintf("LibTopoART v%.2f\n", tac.LibTopoART_version);
            fgetl(file);

            l = fgetl(file);
            fileFormatVersions = sscanf(l,'file format versions: %f; %f');
            tac.fileFormatVersion = fileFormatVersions(1);
            tac.taFileFormatVersion = fileFormatVersions(2);
            assert((tac.fileFormatVersion == 0.01 || tac.fileFormatVersion == 1.00) && ...
                (tac.taFileFormatVersion == 0.10 || tac.taFileFormatVersion == 0.11 || ...
                tac.taFileFormatVersion == 1.00), 'File format version not supported')
        end

        function ReadAdditionalParams(tac, file)
            arguments
                tac TopoARTC
                file
            end

            if tac.fileFormatVersion >= 1.00
                l = fgetl(file);
                tac.nu = sscanf(l, 'nu: %i');
                fgetl(file);
            end
        end

        function ReadModules(tac, file)
            arguments
                tac TopoARTC
                file
            end
            
            for i = 1:tac.moduleNum
                readModules(i) = TopoARTCModule(file, i, tac.longName);
            end
            tac.modules = readModules;
        end
    end

    methods
        function PlotNetwork(tac, colors)
            arguments
                tac TopoARTC
                colors
            end

            for m = 1:tac.moduleNum
                tac.PlotModule(m, colors)
                tac.PlotModule(m, colors, 0, 0, '', ' (class IDs)', '_class_IDs', @tac.GetClassColor, @tac.GetClassColorNum)
            end
        end
    end
end