classdef HypersphereTopoARTC < HypersphereTopoART
    properties
        baseFileFormatVersion {mustBeNumeric}
        nu {mustBeInteger, mustBePositive}
    end
    
    methods (Access = protected)
        function colorIndex = GetClassColor(htac, moduleIndex, nodeIndex)
            arguments
                htac HypersphereTopoARTC
                moduleIndex {mustBeInteger, mustBePositive}
                nodeIndex {mustBeInteger, mustBePositive}
            end
            
            colorIndex = htac.modules(moduleIndex).nodes(nodeIndex).classID - htac.modules(moduleIndex).minClassID + 1;
        end
        
        function colorNum = GetClassColorNum(htac, moduleIndex)
            arguments
                htac HypersphereTopoARTC
                moduleIndex {mustBeInteger, mustBePositive}
            end
            
            colorNum = max(htac.modules(moduleIndex).maxClassID - htac.modules(moduleIndex).minClassID + 1, 2);
        end
        
        function Init(htac)
            htac.shortName = 'HTAC';
            htac.longName = 'Hypersphere TopoART-C';
        end
        
        function file = OpenFile(htac, path)
            arguments
                htac HypersphereTopoARTC
                path char {mustBeFile}
            end
            
            disp(['Load ' htac.longName ' network: ' path])
           
            file = fopen(path,'r');
            assert(file ~= -1, 'File not found')
           
            fgetl(file);
            l = fgetl(file);
            assert(strcmp(l, '* Hypersphere TopoART-C network *') ~= 0, 'No Hypersphere TopoART-C text file')    
            l = fgetl(file);
            htac.LibTopoART_version = sscanf(l(8:length(l)),'LibTopoART (v%f)');
            fprintf("LibTopoART v%.2f\n", htac.LibTopoART_version);
            fgetl(file);
            
            l = fgetl(file);
            fileFormatVersions = sscanf(l,'file format versions: %f; %f; %f');
            htac.fileFormatVersion = fileFormatVersions(1);
            htac.baseFileFormatVersion = fileFormatVersions(2);
            htac.taFileFormatVersion = fileFormatVersions(3);
            assert((htac.fileFormatVersion == 0.01 || htac.fileFormatVersion == 1.00) && ...
                (htac.baseFileFormatVersion == 0.01 || htac.baseFileFormatVersion == 1.00) && ...
                (htac.taFileFormatVersion == 0.10 || htac.taFileFormatVersion == 0.11 || ...
                htac.taFileFormatVersion == 1.00), 'File format version not supported')
        end
        
        function ReadAdditionalParams(htac, file)
            arguments
                htac HypersphereTopoARTC
                file
            end
     
            ReadAdditionalParams@HypersphereTopoART(htac, file)

            if htac.fileFormatVersion >= 1.00
                l = fgetl(file);
                htac.nu = sscanf(l, 'nu: %i');
                fgetl(file);
            end
        end

        function ReadModules(htac, file)
            arguments
                htac HypersphereTopoARTC
                file
            end
            
            for i = 1:htac.moduleNum
                readModules(i) = TopoARTCModule(file, i, htac.longName);
            end
            htac.modules = readModules;
        end
    end
    
    methods
        function PlotNetwork(htac, colors)
            arguments
                htac HypersphereTopoARTC
                colors
            end
            
            for m = 1:htac.moduleNum
                htac.PlotModule(m, colors)
                htac.PlotModule(m, colors, 0, 0, '', ' (class IDs)', '_class_IDs', @htac.GetClassColor, @htac.GetClassColorNum)
            end
        end
    end
end