classdef HypersphereTopoART < TopoART
    properties
        fileFormatVersion {mustBeNumeric}
        R {mustBeNumeric}
    end

    methods (Access = protected)
        function suffix = GetTitleSuffix(hta)
            arguments
                hta HypersphereTopoART
            end
            
            suffix = [', $\bar{R} = ' num2str(hta.R, '%.3f') '$'];
        end 

        function Init(hta)
            hta.shortName = 'HTA';
            hta.longName = 'Hypersphere TopoART';
        end

        function file = OpenFile(hta, path)
            arguments
                hta HypersphereTopoART
                path char {mustBeFile}
            end

            disp(['Load ' hta.longName ' network: ' path])

            file = fopen(path,'r');
            assert(file ~= -1, 'File not found')

            fgetl(file);
            l = fgetl(file);
            assert(strcmp(l, '*  Hypersphere TopoART network  *') ~= 0, 'No Hypersphere TopoART text file')    
            l = fgetl(file);
            hta.LibTopoART_version = sscanf(l(8:length(l)),'LibTopoART (v%f)');
            fprintf("LibTopoART v%.2f\n", hta.LibTopoART_version);
            fgetl(file);

            l = fgetl(file);
            fileFormatVersions = sscanf(l,'file format versions: %f; %f');
            hta.fileFormatVersion = fileFormatVersions(1);
            hta.taFileFormatVersion = fileFormatVersions(2);
            assert((hta.fileFormatVersion == 0.01 || hta.fileFormatVersion == 1.00) && ...
                (hta.taFileFormatVersion == 0.10 || hta.taFileFormatVersion == 0.11 || ...
                hta.taFileFormatVersion == 1.00), 'File format version not supported')
        end

        function PlotWeights(hta, weights, color, off1, off2)
            arguments
                hta HypersphereTopoART
                weights {mustBeVector}
                color
                off1 {mustBeInteger, mustBeNonnegative}
                off2 {mustBeInteger, mustBeNonnegative}
            end
            
            x = weights(1 + off1) - weights(3 + off1 + off2);
            y = weights(2 + off1) - weights(3 + off1 + off2);
            w = 2 * weights(3 + off1 + off2);
            h = w;

            rectangle('Position', [x, y, w, h], 'Curvature', [1, 1], ...
                'EdgeColor', 'black', 'FaceColor', color);
        end

        function ReadAdditionalParams(hta, file)
            arguments
                hta HypersphereTopoART
                file
            end

            l = fgetl(file);
            hta.R = sscanf(l, 'R: %f');
            fgetl(file);
        end
    end
end