classdef EpisodicTopoART < TopoART
    properties
        fileFormatVersion {mustBeNumeric}
        t_F0 {mustBeInteger, mustBeNonnegative}
        t_max {mustBeInteger, mustBePositive}
    end

    methods (Access = protected)
        function suffix = GetTitleSuffix(eta)
            arguments
                eta EpisodicTopoART
            end

            suffix = [', $t^{max} = ' num2str(eta.t_max) '$'];
        end 

        function Init(eta)
            eta.shortName = 'ETA';
            eta.longName = 'Episodic TopoART';
        end

        function file = OpenFile(eta, path)
            arguments
                eta EpisodicTopoART
                path char {mustBeFile}
            end

            disp(['Load ' eta.longName ' network: ' path])

            file = fopen(path,'r');
            assert(file ~= -1, 'File not found')

            fgetl(file);
            l = fgetl(file);
            assert(strcmp(l, '*   Episodic TopoART network    *') ~= 0, 'No Episodic TopoART text file')    
            l = fgetl(file);
            eta.LibTopoART_version = sscanf(l(8:length(l)),'LibTopoART (v%f)');
            fprintf("LibTopoART v%.2f\n", eta.LibTopoART_version);
            fgetl(file);

            l = fgetl(file);
            fileFormatVersions = sscanf(l,'file format versions: %f; %f');
            eta.fileFormatVersion = fileFormatVersions(1);
            eta.taFileFormatVersion = fileFormatVersions(2);
            assert((eta.fileFormatVersion == 0.02 || eta.fileFormatVersion == 1.00) && ...
                (eta.taFileFormatVersion == 0.10 || eta.taFileFormatVersion == 0.11 || ...
                eta.taFileFormatVersion == 1.00), 'File format version not supported')
        end

        function ReadAdditionalParams(eta, file)
            arguments
                eta EpisodicTopoART
                file
            end

            l = fgetl(file);
            eta.t_F0 = sscanf(l, 't^F0: %i');
            l = fgetl(file);
            eta.t_max = sscanf(l, 't^max: %i');
            fgetl(file);
        end

        function ReadModules(eta, file)
            arguments
                eta EpisodicTopoART
                file
            end

            for i = 1:eta.moduleNum
                readModules(i) = EpisodicTopoARTModule(file, i, eta.longName);
            end
            eta.modules = readModules;
        end
    end
end